<?php
require_once __DIR__ . '/../../config/helpers.php';

$method = $_SERVER['REQUEST_METHOD'] ?? 'GET';
$uri = $_SERVER['REQUEST_URI'] ?? '/';
$path = parse_url($uri, PHP_URL_PATH);

// Remove prefix up to /api
$pos = strpos($path, '/api/');
$route = $pos !== false ? substr($path, $pos + 4) : $path;
$route = '/' . ltrim($route, '/');
$parts = array_values(array_filter(explode('/', trim($route, '/'))));

function not_found() { json_response(['success'=>false,'message'=>'Not found'], 404); }

$pdo = db();
$cfg = app_config();

try {
    // AUTH
    if ($parts[0] ?? '' === 'auth') {
        $action = $parts[1] ?? '';
        if ($method === 'POST' && $action === 'request-otp') {
            $data = read_json();
            $phone = trim($data['phone'] ?? '');
            $purpose = ($data['purpose'] ?? 'login') === 'register' ? 'register' : 'login';
            if ($phone === '') json_response(['success'=>false,'message'=>'Phone required'], 422);

            $otp = $cfg['DEMO_OTP'];
            $stmt = $pdo->prepare("INSERT INTO otp_codes (phone, otp, purpose, expires_at) VALUES (?,?,?,DATE_ADD(NOW(), INTERVAL 10 MINUTE))");
            $stmt->execute([$phone, $otp, $purpose]);

            json_response(['success'=>true,'message'=>'OTP sent (demo)', 'otp'=>$otp]);
        }

        if ($method === 'POST' && $action === 'register') {
            $data = read_json();
            $name = trim($data['name'] ?? '');
            $phone = trim($data['phone'] ?? '');
            $email = trim($data['email'] ?? '');
            $otp = trim($data['otp'] ?? '');
            if ($name===''||$phone===''||$email===''||$otp==='') json_response(['success'=>false,'message'=>'name, phone, email, otp required'], 422);
            if ($otp !== $cfg['DEMO_OTP']) json_response(['success'=>false,'message'=>'Invalid OTP'], 401);

            // unique phone/email guard
            $stmt = $pdo->prepare("SELECT id FROM vendors WHERE phone=? OR email=? LIMIT 1");
            $stmt->execute([$phone, $email]);
            if ($stmt->fetch()) json_response(['success'=>false,'message'=>'Vendor already exists'], 409);

            // vendor_code
            $next = $pdo->query("SELECT IFNULL(MAX(id),0)+1 AS next_id FROM vendors")->fetch()['next_id'];
            $vendorCode = 'VEN' . str_pad((string)$next, 3, '0', STR_PAD_LEFT);

            $stmt = $pdo->prepare("INSERT INTO vendors (vendor_code, name, phone, email, approval_status, kyc_status, is_active, wallet_balance) VALUES (?,?,?,?, 'pending','not_submitted',0,0)");
            $stmt->execute([$vendorCode, $name, $phone, $email]);
            $vendorId = (int)$pdo->lastInsertId();

            $stmt = $pdo->prepare("INSERT INTO vendor_kyc (vendor_id, status) VALUES (?, 'not_submitted')");
            $stmt->execute([$vendorId]);

            // return basic
            json_response(['success'=>true,'message'=>'Registered. Proceed to KYC.', 'vendor_id'=>$vendorId, 'vendor_code'=>$vendorCode]);
        }

        if ($method === 'POST' && $action === 'login') {
            $data = read_json();
            $phone = trim($data['phone'] ?? '');
            $otp = trim($data['otp'] ?? '');
            if ($phone===''||$otp==='') json_response(['success'=>false,'message'=>'phone and otp required'], 422);
            if ($otp !== $cfg['DEMO_OTP']) json_response(['success'=>false,'message'=>'Invalid OTP'], 401);

            $stmt = $pdo->prepare("SELECT * FROM vendors WHERE phone=? LIMIT 1");
            $stmt->execute([$phone]);
            $vendor = $stmt->fetch();
            if (!$vendor) json_response(['success'=>false,'message'=>'Vendor not found'], 404);

            $token = random_token(64);
            $stmt = $pdo->prepare("INSERT INTO vendor_tokens (vendor_id, token, expires_at) VALUES (?,?, DATE_ADD(NOW(), INTERVAL 7 DAY))");
            $stmt->execute([(int)$vendor['id'], $token]);

            json_response(['success'=>true,'message'=>'Login ok', 'token'=>$token, 'vendor'=>$vendor]);
        }

        not_found();
    }

    // VENDOR
    if (($parts[0] ?? '') === 'vendor') {
        $vendor = require_vendor_auth();
        $action = $parts[1] ?? '';

        if ($method === 'GET' && ($action === '' || $action === 'me')) {
            $stmt = $pdo->prepare("SELECT v.*, k.status AS kyc_record_status, k.admin_notes FROM vendors v LEFT JOIN vendor_kyc k ON k.vendor_id=v.id WHERE v.id=?");
            $stmt->execute([(int)$vendor['id']]);
            json_response(['success'=>true,'vendor'=>$stmt->fetch()]);
        }

        if ($method === 'POST' && $action === 'kyc') {
            // multipart upload
            $vendorId = (int)$vendor['id'];
            $a1 = upload_file('aadhar_front', 'kyc');
            $a2 = upload_file('aadhar_back', 'kyc');
            $p1 = upload_file('pan_front', 'kyc');
            $p2 = upload_file('pan_back', 'kyc');
            $licPhoto = upload_file('license_photo', 'kyc');
            $licenseNumber = trim($_POST['license_number'] ?? '');

            if (!$a1 || !$a2 || !$p1 || !$p2 || !$licPhoto || $licenseNumber==='') {
                json_response(['success'=>false,'message'=>'All KYC files + license_number required'], 422);
            }

            $stmt = $pdo->prepare("UPDATE vendor_kyc SET aadhar_front_path=?, aadhar_back_path=?, pan_front_path=?, pan_back_path=?, license_photo_path=?, license_number=?, status='submitted', submitted_at=NOW() WHERE vendor_id=?");
            $stmt->execute([$a1,$a2,$p1,$p2,$licPhoto,$licenseNumber,$vendorId]);

            $stmt = $pdo->prepare("UPDATE vendors SET kyc_status='submitted', license_number=? WHERE id=?");
            $stmt->execute([$licenseNumber, $vendorId]);

            json_response(['success'=>true,'message'=>'KYC submitted']);
        }

        if ($method === 'PUT' && $action === 'profile') {
            $data = read_json();
            $name = trim($data['name'] ?? $vendor['name']);
            $email = trim($data['email'] ?? $vendor['email']);
            $stmt = $pdo->prepare("UPDATE vendors SET name=?, email=? WHERE id=?");
            $stmt->execute([$name, $email, (int)$vendor['id']]);
            json_response(['success'=>true,'message'=>'Profile updated']);
        }

        if ($method === 'POST' && $action === 'simulate-approval') {
            $stmt = $pdo->prepare("UPDATE vendors SET approval_status='approved', is_active=1 WHERE id=?");
            $stmt->execute([(int)$vendor['id']]);
            json_response(['success'=>true,'message'=>'Vendor activated (demo)']);
        }

        not_found();
    }

    // DASHBOARD
    if (($parts[0] ?? '') === 'dashboard') {
        $vendor = require_vendor_auth();
        if ($method === 'GET' && (($parts[1] ?? '') === 'summary')) {
            $vendorId = (int)$vendor['id'];

            // wallet balance
            $wallet = (int)$pdo->prepare("SELECT wallet_balance FROM vendors WHERE id=?")->execute([$vendorId]) or 0;

            $stmt = $pdo->prepare("SELECT wallet_balance, total_earnings, total_jobs FROM vendors WHERE id=?");
            $stmt->execute([$vendorId]);
            $v = $stmt->fetch();

            $stmt = $pdo->prepare("SELECT IFNULL(SUM(amount),0) AS today_earnings FROM wallet_transactions WHERE vendor_id=? AND type='earning' AND DATE(created_at)=CURDATE()");
            $stmt->execute([$vendorId]);
            $todayE = (int)$stmt->fetch()['today_earnings'];

            $stmt = $pdo->prepare("SELECT COUNT(*) AS c FROM jobs j JOIN leads l ON l.id=j.lead_id WHERE j.vendor_id=? AND j.status IN ('accepted','in_progress') AND l.time_slot LIKE 'Today%'");
            $stmt->execute([$vendorId]);
            $todayVisits = (int)$stmt->fetch()['c'];

            $stmt = $pdo->prepare("SELECT COUNT(*) AS c FROM jobs j JOIN leads l ON l.id=j.lead_id WHERE j.vendor_id=? AND j.status IN ('accepted','in_progress') AND l.time_slot LIKE 'Tomorrow%'");
            $stmt->execute([$vendorId]);
            $tomorrowVisits = (int)$stmt->fetch()['c'];

            $stmt = $pdo->prepare("SELECT j.id, j.completed_at, j.earning_amount, l.title, l.location, l.time_slot FROM jobs j JOIN leads l ON l.id=j.lead_id WHERE j.vendor_id=? AND j.status='completed' ORDER BY j.completed_at DESC LIMIT 5");
            $stmt->execute([$vendorId]);
            $recentCompleted = $stmt->fetchAll();

            json_response(['success'=>true,'data'=>[
                'today_earnings'=>$todayE,
                'wallet_balance'=>(int)($v['wallet_balance'] ?? 0),
                'today_visits'=>$todayVisits,
                'tomorrow_visits'=>$tomorrowVisits,
                'recent_completed_jobs'=>$recentCompleted,
                'total_earnings'=>(int)($v['total_earnings'] ?? 0),
                'total_jobs'=>(int)($v['total_jobs'] ?? 0),
            ]]);
        }
        not_found();
    }

    // LEADS
    if (($parts[0] ?? '') === 'leads') {
        $vendor = require_vendor_auth();
        $vendorId = (int)$vendor['id'];

        if ($method === 'GET' && count($parts) === 1) {
            $stmt = $pdo->query("SELECT * FROM leads WHERE is_available=1 ORDER BY created_at DESC");
            json_response(['success'=>true,'leads'=>$stmt->fetchAll()]);
        }

        if ($method === 'GET' && count($parts) === 2) {
            $leadId = (int)$parts[1];
            $stmt = $pdo->prepare("SELECT * FROM leads WHERE id=?");
            $stmt->execute([$leadId]);
            $lead = $stmt->fetch();
            if (!$lead) json_response(['success'=>false,'message'=>'Lead not found'], 404);
            json_response(['success'=>true,'lead'=>$lead]);
        }

        if ($method === 'POST' && count($parts) === 3 && $parts[2] === 'pick') {
            $leadId = (int)$parts[1];

            // check lead available
            $stmt = $pdo->prepare("SELECT * FROM leads WHERE id=? AND is_available=1 FOR UPDATE");
            $pdo->beginTransaction();
            $stmt->execute([$leadId]);
            $lead = $stmt->fetch();
            if (!$lead) { $pdo->rollBack(); json_response(['success'=>false,'message'=>'Lead not available'], 409); }

            // wallet
            $stmt = $pdo->prepare("SELECT wallet_balance FROM vendors WHERE id=? FOR UPDATE");
            $stmt->execute([$vendorId]);
            $bal = (int)$stmt->fetch()['wallet_balance'];
            $price = (int)$lead['lead_price'];
            if ($bal < $price) { $pdo->rollBack(); json_response(['success'=>false,'message'=>'Insufficient balance. Recharge needed.'], 402); }

            // deduct and transaction
            $stmt = $pdo->prepare("UPDATE vendors SET wallet_balance = wallet_balance - ? WHERE id=?");
            $stmt->execute([$price, $vendorId]);

            $stmt = $pdo->prepare("INSERT INTO wallet_transactions (vendor_id, type, amount, description, ref_type, ref_id) VALUES (?,?,?,?, 'lead', ?)");
            $stmt->execute([$vendorId, 'debit', $price, 'Picked lead #' . $leadId, $leadId]);

            // create job
            $stmt = $pdo->prepare("INSERT INTO jobs (vendor_id, lead_id, status, earning_amount) VALUES (?,?, 'accepted', 500)");
            $stmt->execute([$vendorId, $leadId]);

            // mark lead unavailable
            $stmt = $pdo->prepare("UPDATE leads SET is_available=0 WHERE id=?");
            $stmt->execute([$leadId]);

            $jobId = (int)$pdo->lastInsertId();
            $pdo->commit();

            json_response(['success'=>true,'message'=>'Lead picked','job_id'=>$jobId]);
        }

        not_found();
    }

    // JOBS
    if (($parts[0] ?? '') === 'jobs') {
        $vendor = require_vendor_auth();
        $vendorId = (int)$vendor['id'];

        if ($method === 'GET' && (($parts[1] ?? '') === 'ongoing')) {
            $stmt = $pdo->prepare("SELECT j.*, l.title, l.location, l.time_slot, l.lead_price FROM jobs j JOIN leads l ON l.id=j.lead_id WHERE j.vendor_id=? AND j.status IN ('accepted','in_progress') ORDER BY j.created_at DESC");
            $stmt->execute([$vendorId]);
            json_response(['success'=>true,'jobs'=>$stmt->fetchAll()]);
        }

        if ($method === 'GET' && (($parts[1] ?? '') === 'completed')) {
            $stmt = $pdo->prepare("SELECT j.*, l.title, l.location, l.time_slot FROM jobs j JOIN leads l ON l.id=j.lead_id WHERE j.vendor_id=? AND j.status='completed' ORDER BY j.completed_at DESC");
            $stmt->execute([$vendorId]);
            json_response(['success'=>true,'jobs'=>$stmt->fetchAll()]);
        }

        if ($method === 'POST' && isset($parts[1]) && isset($parts[2]) && $parts[2] === 'start') {
            $jobId = (int)$parts[1];

            $stmt = $pdo->prepare("SELECT * FROM jobs WHERE id=? AND vendor_id=? LIMIT 1");
            $stmt->execute([$jobId, $vendorId]);
            $job = $stmt->fetch();
            if (!$job) json_response(['success'=>false,'message'=>'Job not found'], 404);
            if ($job['status'] !== 'accepted') json_response(['success'=>false,'message'=>'Job cannot be started'], 409);

            $before = upload_file('before_photo', 'jobs');
            if (!$before) json_response(['success'=>false,'message'=>'before_photo required'], 422);

            $stmt = $pdo->prepare("UPDATE jobs SET status='in_progress', before_photo_path=?, started_at=NOW() WHERE id=?");
            $stmt->execute([$before, $jobId]);

            json_response(['success'=>true,'message'=>'Job started']);
        }

        if ($method === 'POST' && isset($parts[1]) && isset($parts[2]) && $parts[2] === 'complete') {
            $jobId = (int)$parts[1];
            $otp = trim($_POST['otp'] ?? '');

            $stmt = $pdo->prepare("SELECT * FROM jobs WHERE id=? AND vendor_id=? LIMIT 1");
            $stmt->execute([$jobId, $vendorId]);
            $job = $stmt->fetch();
            if (!$job) json_response(['success'=>false,'message'=>'Job not found'], 404);
            if ($job['status'] !== 'in_progress') json_response(['success'=>false,'message'=>'Job cannot be completed'], 409);
            if ($otp !== $cfg['COMPLETION_OTP']) json_response(['success'=>false,'message'=>'Invalid completion OTP'], 401);

            $after = upload_file('after_photo', 'jobs');
            if (!$after) json_response(['success'=>false,'message'=>'after_photo required'], 422);

            $pdo->beginTransaction();

            $earning = (int)($job['earning_amount'] ?? 500);

            $stmt = $pdo->prepare("UPDATE jobs SET status='completed', after_photo_path=?, completion_otp_entered=?, completed_at=NOW() WHERE id=?");
            $stmt->execute([$after, $otp, $jobId]);

            $stmt = $pdo->prepare("UPDATE vendors SET wallet_balance = wallet_balance + ?, total_earnings = total_earnings + ?, total_jobs = total_jobs + 1 WHERE id=?");
            $stmt->execute([$earning, $earning, $vendorId]);

            $stmt = $pdo->prepare("INSERT INTO wallet_transactions (vendor_id, type, amount, description, ref_type, ref_id) VALUES (?,?,?,?, 'job', ?)");
            $stmt->execute([$vendorId, 'earning', $earning, 'Job earning credit', $jobId]);

            $pdo->commit();

            json_response(['success'=>true,'message'=>'Job completed', 'earning'=>$earning]);
        }

        not_found();
    }

    // WALLET
    if (($parts[0] ?? '') === 'wallet') {
        $vendor = require_vendor_auth();
        $vendorId = (int)$vendor['id'];

        if ($method === 'GET' && (($parts[1] ?? '') === 'summary')) {
            $stmt = $pdo->prepare("SELECT wallet_balance, total_earnings, total_jobs FROM vendors WHERE id=?");
            $stmt->execute([$vendorId]);
            json_response(['success'=>true,'data'=>$stmt->fetch()]);
        }

        if ($method === 'GET' && (($parts[1] ?? '') === 'history')) {
            $stmt = $pdo->prepare("SELECT * FROM wallet_transactions WHERE vendor_id=? ORDER BY created_at DESC LIMIT 200");
            $stmt->execute([$vendorId]);
            json_response(['success'=>true,'transactions'=>$stmt->fetchAll()]);
        }

        if ($method === 'POST' && (($parts[1] ?? '') === 'recharge')) {
            $data = read_json();
            $amount = (int)($data['amount'] ?? 0);
            if ($amount <= 0) json_response(['success'=>false,'message'=>'amount must be > 0'], 422);

            $pdo->beginTransaction();
            $stmt = $pdo->prepare("UPDATE vendors SET wallet_balance = wallet_balance + ? WHERE id=?");
            $stmt->execute([$amount, $vendorId]);

            $stmt = $pdo->prepare("INSERT INTO wallet_transactions (vendor_id, type, amount, description, ref_type) VALUES (?,?,?,?, 'recharge')");
            $stmt->execute([$vendorId, 'credit', $amount, 'Recharge via demo payment gateway']);

            $pdo->commit();

            json_response(['success'=>true,'message'=>'Recharged', 'amount'=>$amount]);
        }

        not_found();
    }

    not_found();
} catch (Exception $e) {
    if (db()->inTransaction()) db()->rollBack();
    json_response(['success'=>false,'message'=>'Server error','error'=>$e->getMessage()], 500);
}
